
/*:

@target MZ
@plugindesc 立ち絵などの差分のある絵をまとめて表示したりするプラグインです。
@author メロウ紙芝居
@url https://meroukamisibai.com/

@help MK_CharaPicture.js

文章の表示の名前の制御文字:
\CP[キャラの名前:差分:部位(,で複数可能)]:
キャラの名前に合うキャラがいる場合、名前はキャラの名前に置き換えられ、
指定した部位の差分を指定した差分に変更して、立ち絵顔グラも表示します。
差分、部位は省けます。キャラの名前に合うキャラがいない時は、
キャラの名前が名前になります。
例: 
　\CP[miru:普通:表情] → ミル
　\CP[miru:欲情:表情,体] → ミル
　\CP[あぼどん] → あぼどん

@command explanation_chara
@text ---キャラの表示関係---
@desc

@command visible
@text キャラの表示を変更
@desc 指定したキャラの立ち絵の表示を変更します。

@arg chara
@text キャラ
@desc 指定するキャラの名前です。
@type string

@arg visible
@text キャラの表示
@desc キャラを表示しますか？
@on 表示する
@off 表示しない
@type boolean
@default true

@command visibleAllHide
@text 全キャラを非表示にする
@desc 全キャラの立ち絵の非表示にします。

@command partVisible
@text キャラの部位の表示を変更
@desc 指定したキャラの部位の立ち絵の表示を変更します。

@arg chara
@text キャラ
@desc 指定するキャラの名前です。
@type string

@arg part
@text 部位
@desc 指定する部位の名前です。
@type string

@arg visible
@text 部位の表示
@desc キャラを表示しますか？
@on 表示する
@off 表示しない
@type boolean
@default true

@command change
@text キャラの表示する差分を変更
@desc 指定したキャラの差分を設定します。

@arg chara
@text キャラ
@desc 指定するキャラの名前です。
@type string

@arg difference
@text 差分
@desc 指定する差分の名前です。消すときは何も入力しません。
@type string

@arg partsName
@text 部位
@desc 指定する部位の名前です。未入力の場合、全ての部位が対象になります。「,」で複数の部位を指定できます（例 表情,服）。
@type string

@command name
@text キャラの名前を変更
@desc 指定したキャラの名前を変更します。

@arg chara
@text キャラ
@desc 指定するキャラの名前です。
@type string

@arg name
@text 名前
@desc 入力した名前に変更します。
@type string

@command position
@text キャラの位置を変更
@desc 指定したキャラの位置を変更します。

@arg chara
@text キャラ
@desc 指定するキャラの名前です。
@type string

@arg x
@text x
@desc 表示するx座標です。
@type number
@default 0

@arg y
@text y
@desc 表示するy座標です。
@type number
@default 0

@command position_x
@text キャラのx座標を変更
@desc 指定したキャラのx座標を変更します。

@arg chara
@text キャラ
@desc 指定するキャラの名前です。
@type string

@arg x
@text x
@desc 表示するx座標です。
@type number
@default 0

@command position_y
@text キャラのy座標を変更
@desc 指定したキャラのy座標を変更します。

@arg chara
@text キャラ
@desc 指定するキャラの名前です。
@type string

@arg y
@text y
@desc 表示するy座標です。
@type number
@default 0

@command partPosition
@text キャラの部位の位置を変更
@desc 指定したキャラの部位の位置を変更します。

@arg chara
@text キャラ
@desc 指定するキャラの名前です。
@type string

@arg part
@text 部位
@desc 指定する部位の名前です。
@type string

@arg x
@text x
@desc 表示するx座標です。
@type number
@default 0

@arg y
@text y
@desc 表示するy座標です。
@type number
@default 0

@command scale
@text キャラの拡大率を変更
@desc 指定したキャラの拡大率を変更します。

@arg chara
@text キャラ
@desc 指定するキャラの名前です。
@type string

@arg scale_x
@text 幅
@desc 横の拡大率です。
@type number
@default 100

@arg scale_y
@text 高さ
@desc 縦の拡大率です。
@type number
@default 100

@command opacity
@text キャラの不透明度を変更
@desc 指定したキャラの不透明度を変更します。

@arg chara
@text キャラ
@desc 指定するキャラの名前です。
@type string

@arg opacity
@text 不透明度
@desc 立ち絵の不透明度です。（0～255）
@type number
@max 255
@min 0
@default 255

@command partOpacity
@text キャラの部位の不透明度を変更
@desc 指定したキャラの部位の不透明度を変更します。

@arg chara
@text キャラ
@desc 指定するキャラの名前です。
@type string

@arg part
@text 部位
@desc 指定する部位の名前です。
@type string

@arg opacity
@text 不透明度
@desc 立ち絵の不透明度です。（0～255）
@type number
@max 255
@min 0
@default 255

@command hue
@text キャラの色相を変更
@desc 指定したキャラの色相を変更します。※ウィンドウ非対応

@arg chara
@text キャラ
@desc 指定するキャラの名前です。
@type string

@arg hue
@text 色相
@desc 立ち絵の色相です。（0～360）
@type number
@max 360
@min 0
@default 0

@command partHue
@text キャラの部位の色相を変更
@desc 指定したキャラの部位の色相を変更します。※ウィンドウ非対応

@arg chara
@text キャラ
@desc 指定するキャラの名前です。
@type string

@arg part
@text 部位
@desc 指定する部位の名前です。
@type string

@arg hue
@text 色相
@desc 立ち絵の色相です。（0～360）
@type number
@max 360
@min 0
@default 0

@command explanation_actor
@text ---アクター関係---
@desc

@command actorLinking
@text キャラとアクターの紐づけを変更
@desc 指定したキャラとアクターの紐づけを変更します。

@arg chara
@text キャラ
@desc 指定するキャラの名前です。
@type string

@arg actorId
@text アクター
@desc キャラと紐づけるアクターです。
@type actor
@default 0

@command explanation_face
@text ---立ち絵の顔グラ関係---
@desc

@command faceVisible
@text キャラの立ち絵顔グラの表示を変更
@desc 指定したキャラの立ち絵顔グラの表示を変更します。

@arg chara
@text キャラ
@desc 指定するキャラの名前です。
@type string

@arg visible
@text 立ち絵顔グラの表示
@desc 立ち絵顔グラを表示しますか？
@on 表示する
@off 表示しない
@type boolean
@default true

@command faceVisibleAll
@text 全キャラの立ち絵顔グラの表示を変更
@desc 全キャラの立ち絵顔グラの表示を変更します。

@arg visible
@text 立ち絵顔グラの表示
@desc 立ち絵顔グラを表示しますか？
@on 表示する
@off 表示しない
@type boolean
@default true

@command faceSettings
@text キャラの立ち絵顔グラ設定を変更
@desc 指定したキャラの立ち絵顔グラの設定を変更します。

@arg chara
@text キャラ
@desc 指定するキャラの名前です。
@type string

@arg x
@text x
@desc 表示する立ち絵の横の開始位置です。
@type number
@default 0

@arg y
@text y
@desc 表示する立ち絵の縦の開始位置です。
@type number
@default 0

@arg width
@text 幅
@desc 表示する立ち絵の幅です。
@type number
@default 144

@arg height
@text 高さ
@desc 表示する立ち絵の高さです。
@type number
@default 144

@command explanation_autoOperation
@text ---自動操作関係---
@desc

@command autoOperationActive
@text キャラの自動操作の有効を変更
@desc 指定したキャラの自動操作の有効を変更します。

@arg chara
@text キャラ
@desc 指定するキャラの名前です。
@type string

@arg active
@text 自動操作の有効
@desc 自動操作を有効にしますか？
@on 有効にする
@off 有効にしない
@type boolean
@default true

@command explanation_other
@text ---その他---
@desc

@command preload
@text 立ち絵の画像をプリロード
@desc 立ち絵に使う画像をプリロードしますが、プリロードは自動で行われるので今は必要のない機能です。

*/

/////////////////模倣データベース/////////////////

var mk_charaPicture_data = [null];

(() => {
    
    //----------模倣データベースオブジェクト----------
    
    let data = new MK_CharaPicture_Data();

    //入力オブジェクト
    function MK_CharaPicture_Data() {
        this.chara = null;
        this.id_chara = 0;
        this.id_autoOperation = 0;
        this.id_part = 0;
        this.id_difference = 0;
        this.id_bitmap = 0;
    }

    //キャラ
    MK_CharaPicture_Data.prototype.Chara = function(name) {//名前
        this.id_chara++;
        this.id_autoOperation = 0;
        this.id_part = 0;
        this.id_difference = 0;
        this.id_bitmap = 0;
        mk_charaPicture_data[this.id_chara] = {
            id : this.id_chara,
            name : name,//名前 文字列
            face : {//顔グラ表示設定
                visible : false,//表示するか
                x : 0,
                y : 0,
                width : 144,
                height : 144
            },
            actorLinking : -1,//アクターと立ち絵の紐づけ
            autoOperationActive : false,
            autoOperations : [null],//自動操作オブジェクト
            parts : [null]//部位オブジェクト
        };
        this.chara = mk_charaPicture_data[this.id_chara];
    };
    
    //顔グラ表示設定
    MK_CharaPicture_Data.prototype.Face = function(visible, x, y, width, height) {
        this.chara.face.visible = visible;
        this.chara.face.x = x || 0;
        this.chara.face.y = y || 0;
        this.chara.face.width = width || 144;
        this.chara.face.height = height || 144;
    };
    
    //アクターと立ち絵を紐づける
    MK_CharaPicture_Data.prototype.ActorLinking = function(actoesId) {//アクターid,
        this.chara.actorLinking = actoesId;
    };
    
    //自動操作有効にするか
    MK_CharaPicture_Data.prototype.AutoOperationActive = function(visible) {//trueかfalse
        this.chara.autoOperationActive = visible;
    };

    //自動操作
    MK_CharaPicture_Data.prototype.AutoOperation = function(condition, action) {//js(trueかfalseを返す), js
        this.id_autoOperation++;
        this.chara.autoOperations[this.id_autoOperation] = {
            id : this.id_autoOperation,
            condition : condition || "true",//条件 js(trueかfalseを返す)
            action : action || ""//行動 js
        };
    };

    //部位
    MK_CharaPicture_Data.prototype.Part = function(name) {//名前
        this.id_part++;
        this.id_difference = 0;
        this.chara.parts[this.id_part] = {
            id : this.id_part,
            name : name,//名前 文字列
            differences : [null]//差分オブジェクト
        };
    };

    //差分
    MK_CharaPicture_Data.prototype.Difference = function(name) {//名前
        this.id_difference++;
        this.id_bitmap = 0;
        this.chara.parts[this.id_part].differences[this.id_difference] = {
            id : this.id_difference,
            name : name,//名前 文字列
            bitmaps : [null]//返す画像オブジェクト
        };
    };

    //返す画像
    MK_CharaPicture_Data.prototype.Bitmap = function(name, condition) {//画像名, js(trueかfalseを返す)
        this.id_bitmap++;
        this.chara.parts[this.id_part].differences[this.id_difference].bitmaps[this.id_bitmap] =  {
            id : this.id_bitmap,
            bitmap : name,//返す画像名 文字列
            condition : condition　|| "true"//条件 js(trueかfalseを返す)
        };
    };

    //----------模倣データベース入力場所----------
    
    //-----ここから

    data.Chara("ダブルピース");//-----
    data.Face(true, 115, 70, 201, 205);
    data.ActorLinking(1);

    data.Part("体");
    data.Difference("体_普通");
    data.Bitmap("daburupisu_karada_hutuu");
    data.Difference("体_赤面");
    data.Bitmap("daburupisu_karada_sekimen");
    
    data.Part("服");
    data.Difference("服_私服");
    data.Bitmap("daburupisu_huku_hutuu");
    data.Bitmap("daburupisu_huku_sukurumizugi","$gameVariables.value(22) == 1");
    data.Bitmap("daburupisu_huku_zenra","$gameVariables.value(22) == 2");
    
    data.Part("表情");
    data.Difference("普通");
    data.Bitmap("daburupisu_hyouzyou_hutuu");
    data.Bitmap("daburupisu_hyouzyou_hutuu_reipu","$gameSwitches.value(2)");
    data.Difference("笑う");
    data.Bitmap("daburupisu_hyouzyou_warau");
    data.Bitmap("daburupisu_hyouzyou_warau_reipu","$gameSwitches.value(2)");
    data.Difference("閉じ");
    data.Bitmap("daburupisu_hyouzyou_tozi");
    data.Difference("幸せ");
    data.Bitmap("daburupisu_hyouzyou_siawase");
    data.Difference("困る");
    data.Bitmap("daburupisu_hyouzyou_komaru");
    data.Bitmap("daburupisu_hyouzyou_komaru_reipu","$gameSwitches.value(2)");
    data.Difference("切ない");
    data.Bitmap("daburupisu_hyouzyou_setunai");
    data.Difference("悲しい");
    data.Bitmap("daburupisu_hyouzyou_kanasii");
    data.Bitmap("daburupisu_hyouzyou_kanasii_reipu","$gameSwitches.value(2)");
    data.Difference("泣く");
    data.Bitmap("daburupisu_hyouzyou_naku");
    data.Difference("不機嫌");
    data.Bitmap("daburupisu_hyouzyou_hukigen");
    data.Bitmap("daburupisu_hyouzyou_hukigen_reipu","$gameSwitches.value(2)");
    data.Difference("怒り");
    data.Bitmap("daburupisu_hyouzyou_ikari");
    data.Bitmap("daburupisu_hyouzyou_ikari_reipu","$gameSwitches.value(2)");
    data.Difference("びっくり");
    data.Bitmap("daburupisu_hyouzyou_biltukuri");
    data.Bitmap("daburupisu_hyouzyou_biltukuri_reipu","$gameSwitches.value(2)");
    data.Difference("キラキラ");
    data.Bitmap("daburupisu_hyouzyou_kirakira");
    data.Bitmap("daburupisu_hyouzyou_kirakira_reipu","$gameSwitches.value(2)");
    data.Difference("疑問");
    data.Bitmap("daburupisu_hyouzyou_gimon");
    data.Difference("ギャグ");
    data.Bitmap("daburupisu_hyouzyou_gyagu");
    data.Difference("不機嫌悲しい");
    data.Bitmap("daburupisu_hyouzyou_hukigenNaki");
    data.Bitmap("daburupisu_hyouzyou_hukigenNaki_reipu","$gameSwitches.value(2)");
    data.Difference("怒り泣き");
    data.Bitmap("daburupisu_hyouzyou_ikariNaki");
    data.Bitmap("daburupisu_hyouzyou_ikariNaki_reipu","$gameSwitches.value(2)");
    data.Difference("欲情");
    data.Bitmap("daburupisu_hyouzyou_yokuzyou");
    data.Bitmap("daburupisu_hyouzyou_yokuzyou_reipu","$gameSwitches.value(2)");
    
    data.Part("触手床");
    data.Difference("その他_触手");
    data.Bitmap("daburupisu_syokusyu_on");
    
    data.Part("目的の物ゲット");
    data.Difference("その他_目的の物ゲット");
    data.Bitmap("daburupisu_get_on");

    //ダブルピース------
    
    //-----ここまで

})();


/////////////////立ち絵データオブジェクト/////////////

function MK_CharaPicture_Chara(chara) {//模倣データベースのキャラ
    this.id = chara.id;
    this.name = chara.name;//名前
    this.actorLinking = chara.actorLinking;//アクターと立ち絵の紐づけ
    this.parts = this.PartLoad(chara.parts);//立ち絵部位オブジェクト ここにあるのが表示される。部位登録して追加。表示順は登録順
    this.autoOperationActive = chara.autoOperationActive;
    //スプライト情報
    this.visible = false;//表示するか
    this.x = 0;//位置
    this.y = 0;//位置
    this.scale_x = 100;
    this.scale_y = 100;
    this.opacity = 255;//不透明度
    this.hue = 0;//色相
    this.face = {//顔グラ表示設定
        visible : chara.face.visible,//表示するか
        x : chara.face.x,
        y : chara.face.y,
        width : chara.face.width,
        height : chara.face.height
    };
}

//データベースの部位データ読込
MK_CharaPicture_Chara.prototype.PartLoad = function(parts) {//模倣データベースのキャラの部位
    let p = [null];
    for(let r = 1; r < parts.length; r++){
        p[r] = this.PartAdd(parts[r]);
    }
    return p;
};

//部位データ追加
MK_CharaPicture_Chara.prototype.PartAdd = function(part) {//模倣データベースのキャラの部位
    return {
        id : part.id,
        name : part.name,//部位名
        differenceId : -1,//表示差分id
        visible : true,//表示するか
        x : 0,//位置
        y : 0,//位置
        opacity : 255,//不透明度
        hue : 0,//色相
    };
};

//----------キャラ表示関係----------

//表示切替
MK_CharaPicture_Chara.prototype.Visible = function(visible) {//true（表示）かfalse（非表示）
    this.visible = visible;
};

//指定した部位の表示切替
MK_CharaPicture_Chara.prototype.PartVisible = function(index,visible) {//部位id,部位,true（表示）かfalse（非表示）
    this.parts[index].visible = visible;
};

//表示する立ち絵の差分を変更する
MK_CharaPicture_Chara.prototype.Difference = function(partId,differenceId) {//部位id, 差分id（-1で表示しない）
    this.parts[partId].differenceId = differenceId;
};

//名前を変更する
MK_CharaPicture_Chara.prototype.Name = function(name) {//名前
    this.name = name;
};

//位置を変更
MK_CharaPicture_Chara.prototype.Position = function(x,y) {//x,y
    this.x = x;
    this.y = y;
};

//xを変更
MK_CharaPicture_Chara.prototype.Position_x = function(x) {//x
    this.x = x;
};

//yを変更
MK_CharaPicture_Chara.prototype.Position_y = function(y) {//y
    this.y = y;
};

//部位の位置を変更
MK_CharaPicture_Chara.prototype.PartPosition = function(index,x,y) {//部位id,x,y
    this.parts[index].x = x;
    this.parts[index].y = y;
};

//拡大率を変更
MK_CharaPicture_Chara.prototype.Scale = function(scale_x,scale_y) {//拡大率x,y
    this.scale_x = scale_x;
    this.scale_y = scale_y;
};

//不透明度を変更
MK_CharaPicture_Chara.prototype.Opacity = function(opacity) {//不透明度
    this.opacity = opacity;
};

//部位の不透明度を変更
MK_CharaPicture_Chara.prototype.PartOpacity = function(index,opacity) {//部位id,不透明度
    this.parts[index].opacity = opacity;
};

//色相を変更
MK_CharaPicture_Chara.prototype.Hue = function(hue) {//色相
    this.hue = hue;
};

//部位を指定した色相を変更
MK_CharaPicture_Chara.prototype.PartHue = function(index,hue) {//部位id,色相
    this.parts[index].hue = hue;
};

//----------アクター関係----------

//アクターと立ち絵を紐づける
MK_CharaPicture_Chara.prototype.ActorLinking = function(actoesId) {//アクターid
    this.actorLinking = actoesId;
};

//----------顔グラ関係----------

//顔グラキャラの表示切替
MK_CharaPicture_Chara.prototype.FaceVisible = function(visible) {//true（表示）かfalse（非表示）
    this.face.visible = visible;
};

//立ち絵で顔グラ表示の設定
MK_CharaPicture_Chara.prototype.FaceSettings = function(x,y,width,height) {//位置x,y,幅,高さ
    this.face.x = x;
    this.face.y = y;
    this.face.width = width;
    this.face.height = height;
};

//----------自動操作関係----------

//自動操作を有効にするか
MK_CharaPicture_Chara.prototype.AutoOperationActive = function(active) {//trueかfalse
    this.autoOperationActive = active;
};


/////////////////立ち絵ゲームデータ/////////////////

var mk_charaPicture_game = null;//これに立ち絵情報が入る

//立ち絵データ
function MK_CharaPicture_Game() {
    this.charas = [null];
    for(let r = 1; r < mk_charaPicture_data.length; r++){
        this.charas[r] = new MK_CharaPicture_Chara(mk_charaPicture_data[r]);
    }
    this.messageCharaId = -1;//ここのキャラidがメッセージの顔グラになる。-1でなし
}

//----------キャラ表示関係----------

//全キャラの非表示
MK_CharaPicture_Game.prototype.VisibleAllHide = function() {
    for(let r = 1; r < this.charas.length; r++){
        this.charas[r].visible = false;
    }
};

//----------顔グラ関係----------

//全顔グラキャラの表示切替
MK_CharaPicture_Game.prototype.FaceVisibleAll = function(visible) {//true（表示）かfalse（非表示）
    for(let r = 1; r < this.charas.length; r++){
        this.charas[r].face.visible = visible;
    }
};

//----------自動操作関係----------

//自動操作処理
MK_CharaPicture_Game.prototype.ProcessingAutoOperation = function(index) {//キャラid
    if(mk_charaPicture_game.charas[index].autoOperationActive){
        let chara = mk_charaPicture_game.charas[index];//eval用
        const autoOperations = mk_charaPicture_data[index].autoOperations;
        for(let r = 1;r < autoOperations.length;r++){
            if(eval(autoOperations[r].condition)){
                eval(autoOperations[r].action);
            }
        }
    }
};

//全キャラ自動操作処理
MK_CharaPicture_Game.prototype.AllProcessingAutoOperation = function() {
    for(let r = 1;r < this.charas.length;r++){
        this.ProcessingAutoOperation(this.charas[r].id);
    }
};

//----------画像関係----------

//プリロード
MK_CharaPicture_Game.prototype.Preload = function() {
    let chara = null;
    let difference = null;
    let bitmap = "";
    for(let r1 = 1; r1 < this.charas.length; r1++){
        chara = this.charas[r1];
        for(let r2 = 1; r2 < chara.parts.length; r2++){
            difference = mk_charaPicture_data[chara.id].parts[r2].differences;
            for(let r3 = 1; r3 < difference.length; r3++){
                for(let r4 = 1; r4 < difference[r3].bitmaps.length; r4++){
                    this.LoadCharaPicture(difference[r3].bitmaps[r4].bitmap);
                }
            }
        }
    }
};

//表示する画像を後ろから判定して渡す
MK_CharaPicture_Game.prototype.DifferenceBitmap = function(charaId, partId, differenceId) {//キャラid, 部位id, 差分id
    if(differenceId != -1){
        const bitmaps = mk_charaPicture_data[charaId].parts[partId].differences[differenceId].bitmaps;
        for(let r = bitmaps.length - 1; r > 0; r--){
            if(eval(bitmaps[r].condition)){
                return bitmaps[r].bitmap;
            }
        }
    }
    return "";
};

//imgにあるMK_charaPictureのファイルを読込ます
MK_CharaPicture_Game.prototype.LoadCharaPicture = function(filename) {
    return ImageManager.loadBitmap('img/mk_charaPicture/', filename);
};

//----------検索----------

//名前検索にヒットしたキャラidを返す。なければ-1
MK_CharaPicture_Game.prototype.SearchCharaName = function(name) {//検索するキャラ名
    return mk_charaPicture_data.findIndex((chara) => !!chara && chara.name == name);
};

//名前検索にヒットしたキャラの部位idを返す。なければ-1
MK_CharaPicture_Game.prototype.SearchPartName = function(charaId, name) {//キャラid,検索する部位名
    return mk_charaPicture_data[charaId].parts.findIndex((part) => !!part && part.name == name);
};

//名前検索にヒットしたキャラの差分idを返す。なければ-1
MK_CharaPicture_Game.prototype.SearchDifferenceName = function(charaId, partId, name) {//キャラid,部位id,検索する部位名
    return mk_charaPicture_data[charaId].parts[partId].differences.findIndex((differences) => !!differences && differences.name == name);
};

//紐づけたアクターIDにヒットしたキャラidを返す。なければ-1
MK_CharaPicture_Game.prototype.SearchActorLinking = function(actorsId) {//アクターID
    return this.charas.findIndex((chara) => !!chara && chara.actorLinking == actorsId);
};


//----------更新----------

//セーブデータをロードした時にデータベースとキャラや部位が違う時に修正
MK_CharaPicture_Game.prototype.GameUpdate = function() {
    //キャラ
    if(this.charas.length != mk_charaPicture_data.length){
        if(this.charas.length < mk_charaPicture_data.length){
            for(let r = this.charas.length; r < mk_charaPicture_data.length; r++){
                this.charas[r] = new MK_CharaPicture_Chara(mk_charaPicture_data[r]);
            }
        }else{
            while(this.charas.length > mk_charaPicture_data.length){
                this.charas.pop();
            }
        }
    }
    //部位
    for(let r1 = 1; r1 < this.charas.length; r1++){
        if(this.charas[r1].parts.length != mk_charaPicture_data[r1].parts.length){
            if(this.charas[r1].parts.length < mk_charaPicture_data[r1].parts.length){
                for(let r2 = this.charas[r1].parts.length; r2 < mk_charaPicture_data[r1].parts.length; r2++){
                    this.charas[r1].parts[r2] = this.charas[r1].PartAdd(mk_charaPicture_data[r1].parts[r2]);
                }
            }else{
                while(this.charas[r1].parts.length > mk_charaPicture_data[r1].parts.length){
                    this.charas[r1].parts.pop();
                }
            }
        }
    }
};


//----------便利機能----------

//文字列で差分を変更する
MK_CharaPicture_Game.prototype.DifferenceWord = function(charaName, difference, partsName) {//キャラ名,差分名,部位名（「,」で区切って複数指定可能）
    const chara = mk_charaPicture_data[this.SearchCharaName(charaName)];
    let parts = [];
    let differenceId = -1;
    //部位設定
    if(!!partsName && partsName != ""){
        parts = partsName.split(",");
        for(let r = 0; r < parts.length; r++){
            parts[r] = this.SearchPartName(chara.id, parts[r]);
        }
    }else{
        for(let r = 1; r < mk_charaPicture_data[chara.id].parts.length; r++){
            parts.push(this.SearchPartName(chara.id, chara.parts[r].name));
        }
    }
    
    //差分変更
    for(let r = 0; r < parts.length; r++){
        differenceId = this.SearchDifferenceName(chara.id, parts[r], difference);
        if(differenceId != -1){
            mk_charaPicture_game.charas[chara.id].Difference(parts[r], differenceId);
        }else if(difference == ""){
            mk_charaPicture_game.charas[chara.id].Difference(parts[r], -1);
        }
    }
};


/////////////////キャラスプライト/////////////

//----------立ち絵スプライト----------

//立ち絵スプライト
function MK_CharaPicture_Sprite() {
    this.initialize(...arguments);
}

MK_CharaPicture_Sprite.prototype = Object.create(Sprite.prototype);
MK_CharaPicture_Sprite.prototype.constructor = MK_CharaPicture_Sprite;

MK_CharaPicture_Sprite.prototype.initialize = function() {
    mk_charaPicture_game.Preload();
    Sprite.prototype.initialize.call(this);
    this.MK_CharaPicture_PictureRegister();
};   

//mk_charaPicture_dataを取得して、登録
MK_CharaPicture_Sprite.prototype.MK_CharaPicture_PictureRegister = function() {
    let sprite = null;
    for(let r = 1;r < mk_charaPicture_data.length;r++){
        sprite = new MK_CharaPicture_Sprite_Chara();
        sprite.MK_CharaPicture_CharaRegister(mk_charaPicture_data[r]);
        this.addChild(sprite);
    }
};

//----------キャラスプライト----------

//キャラスプライト
function MK_CharaPicture_Sprite_Chara() {
    this.mk_id = 0;
    
    this.initialize(...arguments);
}

MK_CharaPicture_Sprite_Chara.prototype = Object.create(Sprite.prototype);
MK_CharaPicture_Sprite_Chara.prototype.constructor = MK_CharaPicture_Sprite_Chara;

//キャラの部位を登録
MK_CharaPicture_Sprite_Chara.prototype.MK_CharaPicture_CharaRegister = function(chara) {//キャラ
    this.mk_id = chara.id;
    let sprite = null;
    for(let r = 1;r < chara.parts.length;r++){
        sprite = new MK_CharaPicture_Sprite_Parts();
        sprite.mk_chara_id = this.mk_id;
        sprite.mk_part_id = chara.parts[r].id;
        this.addChild(sprite);
    }
};

//常に
MK_CharaPicture_Sprite_Chara.prototype.update = function() {
    Sprite.prototype.update.call(this);
    if(this.MK_CharaPicture_CheckLoad()){
        const chara = mk_charaPicture_game.charas[this.mk_id];
        this.visible = chara.visible;
        if(chara.visible){
            mk_charaPicture_game.ProcessingAutoOperation(chara.id);
            this.x = chara.x;
            this.y = chara.y;
            this.scale.x = chara.scale_x / 100;
            this.scale.y = chara.scale_y / 100;
            this.opacity = chara.opacity;
        }
    }else{
        this.visible = false;
    }
};

//画像読み込めてるか返す
MK_CharaPicture_Sprite_Chara.prototype.MK_CharaPicture_CheckLoad = function() {
    return this.children.every((sprite) => {
        if(!!sprite.bitmap){
            if(sprite.bitmap._loadingState == "loaded" || sprite.bitmap._url == ""){
                return true;
            }
        }
        return false;
    });
};

//----------部位スプライト----------

//部位スプライト
function MK_CharaPicture_Sprite_Parts() {
    this.mk_chara_id = 0;
    this.mk_part_id = 0;
    
    this.initialize(...arguments);
}

MK_CharaPicture_Sprite_Parts.prototype = Object.create(Sprite.prototype);
MK_CharaPicture_Sprite_Parts.prototype.constructor = MK_CharaPicture_Sprite_Parts;

//常に
MK_CharaPicture_Sprite_Parts.prototype.update = function() {
    Sprite.prototype.update.call(this);
    const chara = mk_charaPicture_game.charas[this.mk_chara_id];
    const part = chara.parts[this.mk_part_id];
    this.visible = part.visible;
    if(part.visible){
        this.move(part.x, part.y);
        this.opacity = part.opacity;
        this.setHue(chara.hue + part.hue);
        const bitmap = mk_charaPicture_game.DifferenceBitmap(chara.id, part.id, part.differenceId);
        this.bitmap = mk_charaPicture_game.LoadCharaPicture(bitmap);
    }
};

(() => {
    
    /////////////////プラグイン関係/////////////////
    
    const pluginName = "MK_CharaPicture";
    
    //----------キャラ表示関係----------
    
    //キャラの表示切替
    PluginManager.registerCommand(pluginName, "visible", args => {
        const charaId = mk_charaPicture_game.SearchCharaName(args.chara);
        mk_charaPicture_game.charas[charaId].Visible((args.visible == "true") ? true : false);
    });
    
    //全キャラ非表示
    PluginManager.registerCommand(pluginName, "visibleAllHide", args => {
        mk_charaPicture_game.VisibleAllHide();
    });
    
    //キャラの部位の表示切替
    PluginManager.registerCommand(pluginName, "partVisible", args => {
        const charaId = mk_charaPicture_game.SearchCharaName(args.chara);
        const partId = mk_charaPicture_game.SearchPartName(charaId, args.part);
        mk_charaPicture_game.charas[charaId].PartVisible((args.visible == "true") ? true : false);
    });
    
    //キャラの差分を変更
    PluginManager.registerCommand(pluginName, "change", args => {
        mk_charaPicture_game.DifferenceWord(args.chara, args.difference, args.partsName);
    });
    
    //キャラの名を変更
    PluginManager.registerCommand(pluginName, "name", args => {
        const charaId = mk_charaPicture_game.SearchCharaName(args.chara);
        mk_charaPicture_game.charas[charaId].Name(args.name);
    });
    
    //キャラの位置を変更
    PluginManager.registerCommand(pluginName, "position", args => {
        const charaId = mk_charaPicture_game.SearchCharaName(args.chara);
        mk_charaPicture_game.charas[charaId].Position(Number(args.x), Number(args.y));
    });
    
    //キャラの位置xを変更
    PluginManager.registerCommand(pluginName, "position_x", args => {
        const charaId = mk_charaPicture_game.SearchCharaName(args.chara);
        mk_charaPicture_game.charas[charaId].Position_x(Number(args.x));
    });
    
    //キャラの位置yを変更
    PluginManager.registerCommand(pluginName, "position_y", args => {
        const charaId = mk_charaPicture_game.SearchCharaName(args.chara);
        mk_charaPicture_game.charas[charaId].Position_y(Number(args.y));
    });
    
    //キャラの部位の位置を変更
    PluginManager.registerCommand(pluginName, "partPosition", args => {
        const charaId = mk_charaPicture_game.SearchCharaName(args.chara);
        const partId = mk_charaPicture_game.SearchPartName(charaId, args.part);
        mk_charaPicture_game.charas[charaId].PartPosition(partId, Number(args.x), Number(args.y));
    });
    
    //キャラの拡大率を変更
    PluginManager.registerCommand(pluginName, "scale", args => {
        const charaId = mk_charaPicture_game.SearchCharaName(args.chara);
        mk_charaPicture_game.charas[charaId].Scale(Number(args.scale_x), Number(args.scale_y));
    });
    
    //キャラの不透明度を変更
    PluginManager.registerCommand(pluginName, "opacity", args => {
        const charaId = mk_charaPicture_game.SearchCharaName(args.chara);
        mk_charaPicture_game.charas[charaId].Opacity(Number(args.opacity));
    });
    
    //キャラの部位の不透明度を変更
    PluginManager.registerCommand(pluginName, "partOpacity", args => {
        const charaId = mk_charaPicture_game.SearchCharaName(args.chara);
        const partId = mk_charaPicture_game.SearchPartName(charaId, args.part);
        mk_charaPicture_game.charas[charaId].PartOpacity(partId, Number(args.opacity));
    });
    
    //キャラの色相を変更
    PluginManager.registerCommand(pluginName, "hue", args => {
        const charaId = mk_charaPicture_game.SearchCharaName(args.chara);
        mk_charaPicture_game.charas[charaId].Hue(Number(args.hue));
    });
    
    //キャラの部位の色相を変更
    PluginManager.registerCommand(pluginName, "partHue", args => {
        const charaId = mk_charaPicture_game.SearchCharaName(args.chara);
        const partId = mk_charaPicture_game.SearchPartName(charaId, args.part);
        mk_charaPicture_game.charas[charaId].PartHue(partId, Number(args.hue));
    });
    
    //----------アクター関係----------
    
    //キャラの立ち絵とアクターを紐づける
    PluginManager.registerCommand(pluginName, "actorLinking", args => {
        const charaId = mk_charaPicture_game.SearchCharaName(args.chara);
        mk_charaPicture_game.charas[charaId].ActorLinking(Number(args.actorId));
    });
    
    //----------顔グラ関係----------
    
    //キャラの顔グラの表示切替
    PluginManager.registerCommand(pluginName, "faceVisible", args => {
        const charaId = mk_charaPicture_game.SearchCharaName(args.chara);
        mk_charaPicture_game.charas[charaId].FaceVisible((args.visible == "true") ? true : false);
    });
    
    //全キャラの顔グラの表示切替
    PluginManager.registerCommand(pluginName, "faceVisibleAll", args => {
        mk_charaPicture_game.FaceVisibleAll((args.visible == "true") ? true : false);
    });
    
    //キャラの立ち絵で顔グラの表示設定
    PluginManager.registerCommand(pluginName, "faceSettings", args => {
        const charaId = mk_charaPicture_game.SearchCharaName(args.chara);
        mk_charaPicture_game.charas[charaId].FaceSettings(Number(args.x), Number(args.y), Number(args.width), Number(args.height));
    });
    
    //----------自動操作関係----------
    
    //キャラの自動操作を有効にするか
    PluginManager.registerCommand(pluginName, "autoOperationActive", args => {
        const charaId = mk_charaPicture_game.SearchCharaName(args.chara);
        mk_charaPicture_game.charas[charaId].AutoOperationActive((args.active == "true") ? true : false);
    });
    
    //----------その他関係----------
    
    //プリロード
    PluginManager.registerCommand(pluginName, "preload", args => {
        mk_charaPicture_game.Preload();
    });
    
    
    /////////////////元々オブジェクト/////////////
    
    //ゲーム開始時作成データ
    const _DataManager_createGameObjects = DataManager.createGameObjects;
    DataManager.createGameObjects = function() {
        _DataManager_createGameObjects.apply(this, arguments);
        mk_charaPicture_game = new MK_CharaPicture_Game();
    };
    
    //セーブ
    const _DataManager_makeSaveContents = DataManager.makeSaveContents;
    DataManager.makeSaveContents = function() {
        let contents = _DataManager_makeSaveContents.apply(this, arguments);
        contents.mk_charaPicture = mk_charaPicture_game;
        return contents;
    };
    
    //ロード
    const _DataManager_extractSaveContents = DataManager.extractSaveContents;
    DataManager.extractSaveContents = function(contents) {
        _DataManager_extractSaveContents.apply(this, arguments);
        mk_charaPicture_game = contents.mk_charaPicture;
        mk_charaPicture_game.GameUpdate();
    };
    
    //名前制御文字
    const _Game_Message_setSpeakerName = Game_Message.prototype.setSpeakerName;
    Game_Message.prototype.setSpeakerName = function(speakerName) {
        if(!!speakerName){
            speakerName = speakerName.replace(/\\CP\[(.*?)\]/gi, (_, p1) =>
                this.MK_CharaPicture_EscapeTypeChange(p1)
            );
        }
        _Game_Message_setSpeakerName.apply(this, arguments);
    };
    
    //キャラの画像を変えて、キャラの表示名を返す
    Game_Message.prototype.MK_CharaPicture_EscapeTypeChange = function(name) {
        word = name.split(':');
        const chara = mk_charaPicture_game.charas[mk_charaPicture_game.SearchCharaName(word[0])];
        if(word[1] != null && chara != null){
            mk_charaPicture_game.DifferenceWord(word[0],word[1],word[2]);
        }
        //ついでにメッセージに立ち絵顔グラを設定
        mk_charaPicture_game.messageCharaId = (chara != null && chara.face.visible) ? chara.id : -1;
        return (chara != null) ? chara.name : word[0];
    };
    
    //スプライト作る
    const _Spriteset_Base_createUpperLayer = Spriteset_Base.prototype.createUpperLayer;
    Spriteset_Base.prototype.createUpperLayer = function() {
        this.mk_charaPicture_sprite = new MK_CharaPicture_Sprite();
        this.addChild(this.mk_charaPicture_sprite);
        _Spriteset_Base_createUpperLayer.apply(this, arguments);
    };
    
    //ウィンドウ初期化
    const _Window_Base_initialize = Window_Base.prototype.initialize;
    Window_Base.prototype.initialize = function(rect) {
        _Window_Base_initialize.apply(this, arguments);
        this.mk_charaPicture_bitmapData = [];
    };
    
    //立ち絵表示
    Window_Base.prototype.MK_CharaPicture_DrawCharaPicture = function(charaId, x, y, scale_x, scale_y) {//キャラid,位置x,y,拡大率x,y
        scale_x = scale_x || 100;
        scale_y = scale_y || 100;
        let b = null;
        const charaData = this.MK_CharaPicture_DrawCharaPictureData(charaId);
        const id = this.MK_CharaPicture_BitmapDataID();
        charaData.forEach((data) => {
            b = mk_charaPicture_game.LoadCharaPicture(data.bitmap);
            this.MK_CharaPicture_BitmapData(id, data.bitmap, data.opacity, 0, 0, b.width, b.height, data.x, data.y, b.width * scale_x / 100, b.height * scale_y / 100);
        });
    };
    
    //顔グラサイズで立ち絵表示
    Window_Base.prototype.MK_CharaPicture_DrawCharaPictureFaceSize = function(charaId, x, y, width, height) {//キャラid,位置x,y,幅,高さ（幅と高さは無くてもいい）
        const face = mk_charaPicture_game.charas[charaId].face;
        const charaData = this.MK_CharaPicture_DrawCharaPictureData(charaId);
        width = width || ImageManager.faceWidth;//顔グラサイズ
        height = height || ImageManager.faceHeight;
        const pw = face.width;//元々の立ち絵顔グラサイズ
        const ph = face.height;
        const difference_w = (width - ImageManager.faceWidth) / ImageManager.faceWidth * 100;//指定したサイズは通常顔グラの何%の差があるか
        const difference_h = (height - ImageManager.faceHeight) / ImageManager.faceHeight * 100;
        const sw = Math.min(pw + pw * difference_w / 100, pw);////立ち絵顔グラの幅
        const sh = Math.min(ph + ph * difference_h / 100, ph);
        const dx = Math.floor(x + Math.max(width - ImageManager.faceWidth, 0) / 2);//ウィンドウの表示位置
        const dy = Math.floor(y + Math.max(height - ImageManager.faceHeight, 0) / 2);
        const sx = face.x + Math.max(pw - sw, 0) / 2;//立ち絵顔グラの表示開始位置
        const sy = face.y + Math.max(ph - sh, 0) / 2;
        let scaleWidth = sw;//大きさ
        let scaleHeight = sh;
        if(sw > width || sh > height){//必要なら縮小
            let w = 100;
            let h = 100;
            while(true){
                if(sw * w / 100 <= width && sh * h / 100 <= height){
                    scaleWidth = sw * w / 100;
                    scaleHeight = sh * h / 100;
                    break;
                }
                w--;
                h--;
            }
        }
        let scaleCorrection_x = (width - scaleWidth) / 2;//縮小した時の位置補正
        let scaleCorrection_y = (height - scaleHeight) / 2;
        
        const  id = this.MK_CharaPicture_BitmapDataID();
        charaData.forEach((data) => {
            this.MK_CharaPicture_BitmapData(id, data.bitmap, data.opacity, sx - data.x, sy - data.y, sw, sh, dx + scaleCorrection_x, dy + scaleCorrection_y, scaleWidth, scaleHeight);
        });
    };
    
    //ウィンドウで立ち絵表示するキャラの部位データまとめて返す
    Window_Base.prototype.MK_CharaPicture_DrawCharaPictureData = function(charaId) {//キャラid
        let charaData = [];
        const chara_data = mk_charaPicture_data[charaId];
        const chara_game = mk_charaPicture_game.charas[charaId];
        const parts = chara_game.parts;
        let b = "";
        mk_charaPicture_game.ProcessingAutoOperation(chara_data.id);
        for(let r = 1;r < parts.length; r++){
            if(parts[r].visible){ 
                const data = new function(){
                    this.x = parts[r].x;
                    this.y = parts[r].y;
                    this.bitmap = mk_charaPicture_game.DifferenceBitmap(chara_data.id, parts[r].id, parts[r].differenceId);
                    this.opacity = parts[r].opacity * (chara_game.opacity / 255 * 100) / 100;
                    //this.hue = chara_game.hue + parts[r].hue;//色相の変え方不明なので封印中
                };
                charaData.push(data);
            }
        }
        return charaData;
    };
    
    //立ち絵表示に使用してないidを返す
    Window_Base.prototype.MK_CharaPicture_BitmapDataID = function() {
        let id = 0;
        while(true){
            if(!this.mk_charaPicture_bitmapData[id] || this.mk_charaPicture_bitmapData[id].length == 0){
                break;
            }
            id++;
        }
        this.mk_charaPicture_bitmapData[id] = [];
        return id;
    };
    
    //立ち絵を表示する情報設定
    Window_Base.prototype.MK_CharaPicture_BitmapData = function(id, bitmap, opacity, sx, sy, sw, sh, dx, dy, dw, dh) {//id,画像,不透明度,切り取りx,y,幅x,y,位置x,y,拡大x,y
        const data = {
            bitmap :　mk_charaPicture_game.LoadCharaPicture(bitmap),
            opacity : opacity,
            sx : sx,
            sy : sy,
            sw : sw,
            sh : sh,
            dx : dx,
            dy : dy,
            dw : dw,
            dh : dh
        };
        this.mk_charaPicture_bitmapData[id].push(data);
        data.bitmap.addLoadListener(() => {
            this.MK_CharaPicture_DrawBitmap(id);
        });
    };
    
    //画像全部読み込めてたら立ち絵を表示する
    Window_Base.prototype.MK_CharaPicture_DrawBitmap = function(id) {//id
        if(this.MK_CharaPicture_CheckLoad(id)){
            let data = null;
            this.mk_charaPicture_bitmapData[id].forEach((data) => {
                const momentOpacity = this.contents.paintOpacity;
                this.contents.paintOpacity = data.opacity;
                this.contents.blt(data.bitmap, data.sx, data.sy, data.sw, data.sh, data.dx, data.dy, data.dw, data.dh);//画像,切り取りx,y,幅x,y,位置x,y,拡大x,y
                this.contents.paintOpacity = momentOpacity;
            });
            this.mk_charaPicture_bitmapData[id] = [];
        }
    };
    
    //画像読み込めてるか返す
    Window_Base.prototype.MK_CharaPicture_CheckLoad = function(id) {//id
        return this.mk_charaPicture_bitmapData[id].every((data) => {
            if(!!data.bitmap){
                if(data.bitmap._loadingState == "loaded" || data.bitmap._url == ""){
                    return true;
                }
            }
            return false;
        });
    };
    
    //顔グラ表示
    const _Window_StatusBase_drawActorFace = Window_StatusBase.prototype.drawActorFace;
    Window_StatusBase.prototype.drawActorFace = function(actor, x, y, width, height) {
        const charaId = mk_charaPicture_game.SearchActorLinking(actor._actorId);
        if(charaId != -1){
            this.MK_CharaPicture_DrawCharaPictureFaceSize(charaId, x, y, width, height);
        }
        _Window_StatusBase_drawActorFace.apply(this, arguments);
    };
    
    //メッセージスタートのx位置設定
    const _Window_Message_newLineX = Window_Message.prototype.newLineX;
    Window_Message.prototype.newLineX = function(textState) {
        const x = _Window_Message_newLineX.apply(this, arguments);
        if(mk_charaPicture_game.messageCharaId != -1){
            const faceWidth = ImageManager.faceWidth;
            const spacing = 20;
            const margin = faceWidth + spacing;
            return textState.rtl ? this.innerWidth - margin : margin;
        }
        return x;
    };
    
    //顔グラとか表示するかの判定
    const _Window_Message_drawMessageFace = Window_Message.prototype.drawMessageFace;
    Window_Message.prototype.drawMessageFace = function() {
        if(mk_charaPicture_game.messageCharaId != -1){
            const chara = mk_charaPicture_game.charas[mk_charaPicture_game.messageCharaId];
            if(chara.face.visible){
                const rtl = $gameMessage.isRTL();
                const width = ImageManager.faceWidth;
                const height = this.innerHeight;
                const x = rtl ? this.innerWidth - width - 4 : 4;
                this.MK_CharaPicture_DrawCharaPictureFaceSize(chara.id, x, 0, width, height);
            }
            mk_charaPicture_game.messageCharaId = -1;
        }
        _Window_Message_drawMessageFace.apply(this, arguments);
    };
    
    
})();
